/*
 * Wolfenstein: Enemy Territory GPL Source Code
 * Copyright (C) 1999-2010 id Software LLC, a ZeniMax Media company.
 *
 * ET: Legacy
 * Copyright (C) 2012 Jan Simek <mail@etlegacy.com>
 *
 * This file is part of ET: Legacy - http://www.etlegacy.com
 *
 * ET: Legacy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ET: Legacy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ET: Legacy. If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, Wolfenstein: Enemy Territory GPL Source Code is also
 * subject to certain additional terms. You should have received a copy
 * of these additional terms immediately following the terms and conditions
 * of the GNU General Public License which accompanied the source code.
 * If not, please request a copy in writing from id Software at the address below.
 *
 * id Software LLC, c/o ZeniMax Media Inc., Suite 120, Rockville, Maryland 20850 USA.
 *
 * @file sv_main_Facade.c
 */
#ifdef DEDICATED
#include "server.h"

/*#ifdef TRACKBASE_SUPPORT
#include "sv_trackbase.h"
#endif
*/

static void SVC_Status(netadr_t from, qboolean force);

#define LL(x) x = LittleLong(x)

/*
==============================================================================

CONNECTIONLESS COMMANDS

==============================================================================
*/

typedef struct leakyBucket_s leakyBucket_t;
struct leakyBucket_s
{
	netadrtype_t type;

	union
	{
		byte _4[4];
		byte _6[16];
	} ipv;

	int lastTime;
	signed char burst;

	long hash;

	leakyBucket_t *prev, *next;
};

// This is deliberately quite large to make it more of an effort to DoS
#define MAX_BUCKETS         16384
#define MAX_HASHES          1024

static leakyBucket_t buckets_F[MAX_BUCKETS];
static leakyBucket_t *bucketHashes_F[MAX_HASHES];

static long SVC_HashForAddress_Facade(netadr_t address)
{
	byte   *ip  = NULL;
	size_t size = 0;
	int    i;
	long   hash = 0;

	switch (address.type)
	{
	case NA_IP:  ip = address.ip;  size = 4; break;
	case NA_IP6: ip = address.ip6; size = 16; break;
	default: break;
	}

	for (i = 0; i < size; i++)
	{
		hash += (long)(ip[i]) * (i + 119);
	}

	hash  = (hash ^ (hash >> 10) ^ (hash >> 20));
	hash &= (MAX_HASHES - 1);

	return hash;
}

/*
 * @brief Find or allocate a bucket for an address
 */
static leakyBucket_t *SVC_BucketForAddress_Facade(netadr_t address, int burst, int period)
{
	leakyBucket_t *bucket = NULL;
	int           i;
	long          hash = SVC_HashForAddress_Facade(address);
	int           now  = Sys_Milliseconds();

	for (bucket = bucketHashes_F[hash]; bucket; bucket = bucket->next)
	{
		switch (bucket->type)
		{
		case NA_IP:
			if (memcmp(bucket->ipv._4, address.ip, 4) == 0)
			{
				return bucket;
			}
			break;

		case NA_IP6:
			if (memcmp(bucket->ipv._6, address.ip6, 16) == 0)
			{
				return bucket;
			}
			break;

		default:
			break;
		}
	}

	for (i = 0; i < MAX_BUCKETS; i++)
	{
		int interval;

		bucket   = &buckets_F[i];
		interval = now - bucket->lastTime;

		// Reclaim expired buckets
		if (bucket->lastTime > 0 && (interval > (burst * period) ||
		                             interval < 0))
		{
			if (bucket->prev != NULL)
			{
				bucket->prev->next = bucket->next;
			}
			else
			{
				bucketHashes_F[bucket->hash] = bucket->next;
			}

			if (bucket->next != NULL)
			{
				bucket->next->prev = bucket->prev;
			}

			Com_Memset(bucket, 0, sizeof(leakyBucket_t));
		}

		if (bucket->type == NA_BAD)
		{
			bucket->type = address.type;
			switch (address.type)
			{
			case NA_IP:  Com_Memcpy(bucket->ipv._4, address.ip, 4);   break;
			case NA_IP6: Com_Memcpy(bucket->ipv._6, address.ip6, 16); break;
			default: break;
			}

			bucket->lastTime = now;
			bucket->burst    = 0;
			bucket->hash     = hash;

			// Add to the head of the relevant hash chain
			bucket->next = bucketHashes_F[hash];
			if (bucketHashes_F[hash] != NULL)
			{
				bucketHashes_F[hash]->prev = bucket;
			}

			bucket->prev       = NULL;
			bucketHashes_F[hash] = bucket;

			return bucket;
		}
	}

	// Couldn't allocate a bucket for this address
	return NULL;
}

static qboolean SVC_RateLimit_Facade(leakyBucket_t *bucket, int burst, int period)
{
	if (bucket != NULL)
	{
		int now              = Sys_Milliseconds();
		int interval         = now - bucket->lastTime;
		int expired          = interval / period;
		int expiredRemainder = interval % period;

		if (expired > bucket->burst)
		{
			bucket->burst    = 0;
			bucket->lastTime = now;
		}
		else
		{
			bucket->burst   -= expired;
			bucket->lastTime = now - expiredRemainder;
		}

		if (bucket->burst < burst)
		{
			bucket->burst++;

			return qfalse;
		}
	}

	return qtrue;
}

/*
 * @brief Rate limit for a particular address
 */
static qboolean SVC_RateLimitAddress_Facade(netadr_t from, int burst, int period)
{
	leakyBucket_t *bucket = SVC_BucketForAddress_Facade(from, burst, period);

	return SVC_RateLimit_Facade(bucket, burst, period);
}

/*
 * @brief Send serverinfo cvars, etc to master servers when game complete or
 * by request of getstatus calls.
 * Useful for tracking global player stats.
 * Param force toogles rate limit checks
 */
void SVC_Status_Facade(netadr_t from, qboolean force)
{
	char                 player[1024];
	char                 status[MAX_MSGLEN];
	int                  i;
	client_t             *cl;
	playerState_t        *ps;
	int                  statusLength;
	int                  playerLength;
	char                 infostring[MAX_INFO_STRING];
	static leakyBucket_t bucket;

	if (!force)
	{
		// Prevent using getstatus as an amplifier
		if (SVC_RateLimitAddress_Facade(from, 10, 1000))
		{
			Com_DPrintf("SVC_Status_Facade: rate limit from %s exceeded, dropping request\n",
						NET_AdrToString(from));
			return;
		}

		// Allow getstatus to be DoSed relatively easily, but prevent
		// excess outbound bandwidth usage when being flooded inbound
		if (SVC_RateLimit_Facade(&bucket, 10, 100))
		{
			Com_DPrintf("SVC_Status_Facade: rate limit exceeded, dropping request\n");
			return;
		}
	}
	strcpy(infostring, Cvar_InfoString(CVAR_SERVERINFO | CVAR_SERVERINFO_NOUPDATE));

	// echo back the parameter to status. so master servers can use it as a challenge
	// to prevent timed spoofed reply packets that add ghost servers
	Info_SetValueForKey(infostring, "challenge", Cmd_Argv(1));
	Info_SetValueForKey(infostring, "version", ET_VERSION);

	status[0]    = 0;
	statusLength = 0;

	for (i = 0 ; i < sv_maxclients->integer ; i++)
	{
		cl = &svs.clients[i];
		if (cl->state >= CS_CONNECTED)
		{
			ps = SV_GameClientNum(i);
			Com_sprintf(player, sizeof(player), "%i %i \"%s\"\n",
			            ps->persistant[PERS_SCORE], cl->ping, cl->name);
			playerLength = strlen(player);
			if (statusLength + playerLength >= sizeof(status))
			{
				break;      // can't hold any more
			}
			strcpy(status + statusLength, player);
			statusLength += playerLength;
		}
	}
	
	Info_SetValueForKey( infostring, "protocol", va( "%i", PROTOCOL_VERSION_FACADE) );
	NET_OutOfBandPrint_Facade(NS_SERVER, from, "statusResponse\n%s\n%s", infostring, status);
}

/*
 * @brief Responds with a short info message that should be enough to determine
 * if a user is interested in a server to do a full status
 */
void SVC_Info_Facade(netadr_t from)
{
	int  i, count, humans;
	char *gamedir;
	char infostring[MAX_INFO_STRING];
	char *antilag;
	char *weaprestrict;
	char *balancedteams;

	/*
	 * Check whether Cmd_Argv(1) has a sane length. This was not done in the original Quake3 version which led
	 * to the Infostring bug discovered by Luigi Auriemma. See http://aluigi.altervista.org/ for the advisory.
	 */

	// A maximum challenge length of 128 should be more than plenty.
	if (strlen(Cmd_Argv(1)) > 128)
	{
		return;
	}

	// don't count privateclients
	count = humans = 0;
	for (i = sv_privateClients->integer ; i < sv_maxclients->integer ; i++)
	{
		if (svs.clients[i].state >= CS_CONNECTED)
		{
			count++;
			if (svs.clients[i].netchan.remoteAddress.type != NA_BOT)
			{
				humans++;
			}
		}
	}

	infostring[0] = 0;

	// echo back the parameter to status. so servers can use it as a challenge
	// to prevent timed spoofed reply packets that add ghost servers
	Info_SetValueForKey(infostring, "challenge", Cmd_Argv(1));

	Info_SetValueForKey(infostring, "protocol", va("%i", PROTOCOL_VERSION_FACADE));
	Info_SetValueForKey(infostring, "hostname", sv_hostname->string);
	Info_SetValueForKey(infostring, "serverload", va("%i", svs.serverLoad));
	Info_SetValueForKey(infostring, "mapname", sv_mapname->string);
	Info_SetValueForKey(infostring, "clients", va("%i", count));
	Info_SetValueForKey(infostring, "sv_maxclients", va("%i", sv_maxclients->integer - sv_privateClients->integer));
	Info_SetValueForKey(infostring, "gametype", Cvar_VariableString("g_gametype"));
	Info_SetValueForKey(infostring, "pure", va("%i", sv_pure->integer));

	if (sv_minPing->integer)
	{
		Info_SetValueForKey(infostring, "minPing", va("%i", sv_minPing->integer));
	}
	if (sv_maxPing->integer)
	{
		Info_SetValueForKey(infostring, "maxPing", va("%i", sv_maxPing->integer));
	}
	gamedir = Cvar_VariableString("fs_game");
	if (*gamedir)
	{
		Info_SetValueForKey(infostring, "game", gamedir);
	}
	Info_SetValueForKey(infostring, "sv_allowAnonymous", va("%i", sv_allowAnonymous->integer));

	Info_SetValueForKey(infostring, "friendlyFire", va("%i", sv_friendlyFire->integer));            // NERVE - SMF
	Info_SetValueForKey(infostring, "maxlives", va("%i", sv_maxlives->integer ? 1 : 0));            // NERVE - SMF
	Info_SetValueForKey(infostring, "needpass", va("%i", sv_needpass->integer ? 1 : 0));
	Info_SetValueForKey(infostring, "gamename", GAMENAME_STRING);                                 // Arnout: to be able to filter out Quake servers

	// TTimo
	antilag = Cvar_VariableString("g_antilag");
	if (antilag)
	{
		Info_SetValueForKey(infostring, "g_antilag", antilag);
	}

	weaprestrict = Cvar_VariableString("g_heavyWeaponRestriction");
	if (weaprestrict)
	{
		Info_SetValueForKey(infostring, "weaprestrict", weaprestrict);
	}

	balancedteams = Cvar_VariableString("g_balancedteams");
	if (balancedteams)
	{
		Info_SetValueForKey(infostring, "balancedteams", balancedteams);
	}

	NET_OutOfBandPrint_Facade(NS_SERVER, from, "infoResponse\n%s", infostring);
}

/*
 * DRDoS stands for "Distributed Reflected Denial of Service".
 * See here: http://www.lemuria.org/security/application-drdos.html
 *
 * If the address isn't NA_IP, it's automatically denied.
 *
 * @retval qfalse if we're good.
 * @retval qtrue return value means we need to block.
 */
qboolean SV_CheckDRDoS_Facade(netadr_t from)
{
	int        i;
	int        globalCount;
	int        specificCount;
	receipt_t  *receipt;
	netadr_t   exactFrom;
	int        oldest;
	int        oldestTime;
	static int lastGlobalLogTime   = 0;
	static int lastSpecificLogTime = 0;

	// Usually the network is smart enough to not allow incoming UDP packets
	// with a source address being a spoofed LAN address.  Even if that's not
	// the case, sending packets to other hosts in the LAN is not a big deal.
	// NA_LOOPBACK qualifies as a LAN address.
	if (Sys_IsLANAddress(from))
	{
		return qfalse;
	}

	exactFrom = from;

	if (from.type == NA_IP)
	{
		from.ip[3] = 0; // xx.xx.xx.0
	}
	else
	{
		from.ip6[15] = 0;
	}

	// Count receipts in last 2 seconds.
	globalCount   = 0;
	specificCount = 0;
	receipt       = &svs.infoReceipts_F[0];
	oldest        = 0;
	oldestTime    = 0x7fffffff;
	for (i = 0; i < MAX_INFO_RECEIPTS; i++, receipt++)
	{
		if (receipt->time + 2000 > svs.time)
		{
			if (receipt->time)
			{
				// When the server starts, all receipt times are at zero.  Furthermore,
				// svs.time is close to zero.  We check that the receipt time is already
				// set so that during the first two seconds after server starts, queries
				// from the master servers don't get ignored.  As a consequence a potentially
				// unlimited number of getinfo+getstatus responses may be sent during the
				// first frame of a server's life.
				globalCount++;
			}
			if (NET_CompareBaseAdr(from, receipt->adr))
			{
				specificCount++;
			}
		}
		if (receipt->time < oldestTime)
		{
			oldestTime = receipt->time;
			oldest     = i;
		}
	}

	if (globalCount == MAX_INFO_RECEIPTS)   // All receipts happened in last 2 seconds.
	{
		if (lastGlobalLogTime + 1000 <= svs.time)  // Limit one log every second.
		{
			Com_Printf("Detected flood of getinfo/getstatus connectionless packets\n");
			lastGlobalLogTime = svs.time;
		}
		return qtrue;
	}
	if (specificCount >= 3)   // Already sent 3 to this IP in last 2 seconds.
	{
		if (lastSpecificLogTime + 1000 <= svs.time)   // Limit one log every second.
		{
			Com_Printf("Possible DRDoS attack to address %i.%i.%i.%i, ignoring getinfo/getstatus connectionless packet\n",
			           exactFrom.ip[0], exactFrom.ip[1], exactFrom.ip[2], exactFrom.ip[3]);
			lastSpecificLogTime = svs.time;
		}
		return qtrue;
	}

	receipt       = &svs.infoReceipts_F[oldest];
	receipt->adr  = from;
	receipt->time = svs.time;
	return qfalse;
}


/*
 * A connectionless packet has four leading 0xff characters to distinguish it
 * from a game channel.
 * Clients that are in the game can still send connectionless packets.
 */
static void SV_ConnectionlessPacket_Facade(netadr_t from, msg_t *msg)
{
	char *s;
	char *c;

	MSG_BeginReadingOOB(msg);
	MSG_ReadLong(msg);          // skip the -1 marker

	if (!Q_strncmp("connect", (char *)&msg->data[4], 7))
	{
		Huff_Decompress(msg, 12);
	}

	s = MSG_ReadStringLine(msg);

	Cmd_TokenizeString(s);

	c = Cmd_Argv(0);
	Com_DPrintf("SV packet %s : %s\n", NET_AdrToString(from), c);

	if (!Q_stricmp(c, "getstatus"))
	{
		SVC_Status_Facade(from, qfalse);
	}
	else if (!Q_stricmp(c, "getinfo"))
	{
		SVC_Info_Facade(from);
	}
	else if (!Q_stricmp(c, "getchallenge"))
	{
		SV_GetChallenge(from);
	}
	else if (!Q_stricmp(c, "connect"))
	{
		SV_DirectConnect_Facade(from);
	}/*	7Killer no rcon on facade
	else if (!Q_stricmp(c, "rcon"))
	{
		SVC_RemoteCommand(from, msg);
	}*/
	else if (!Q_stricmp(c, "disconnect"))
	{
		// if a client starts up a local server, we may see some spurious
		// server disconnect messages when their new server sees our final
		// sequenced messages to the old client
	}
	else
	{
		Com_DPrintf("facade bad connectionless packet from %s:\n%s\n"
		            , NET_AdrToString(from), s);
	}
}

void SV_PacketEvent_Facade(netadr_t from, msg_t *msg)
{
	int      i;
	client_t *cl;
	int      qport;

	// check for connectionless packet (0xffffffff) first
	if (msg->cursize >= 4 && *(int *)msg->data == -1)
	{
		SV_ConnectionlessPacket_Facade(from, msg);
		return;
	}

	// read the qport out of the message so we can fix up
	// stupid address translating routers
	MSG_BeginReadingOOB(msg);
	MSG_ReadLong(msg);                  // sequence number
	qport = MSG_ReadShort(msg) & 0xffff;

	// find which client the message is from
	for (i = 0, cl = svs.clients ; i < sv_maxclients->integer ; i++, cl++)
	{
		if (cl->state == CS_FREE)
		{
			continue;
		}
		if (!NET_CompareBaseAdr(from, cl->netchan.remoteAddress))
		{
			continue;
		}
		// it is possible to have multiple clients from a single IP
		// address, so they are differentiated by the qport variable
		if (cl->netchan.qport != qport)
		{
			continue;
		}

		// the IP port can't be used to differentiate them, because
		// some address translating routers periodically change UDP
		// port assignments
		if (cl->netchan.remoteAddress.port != from.port)
		{
			Com_Printf("SV_PacketEvent: fixing up a translated port\n");
			cl->netchan.remoteAddress.port = from.port;
		}

		// make sure it is a valid, in sequence packet
		if (SV_Netchan_Process(cl, msg))
		{
			// zombie clients still need to do the Netchan_Process
			// to make sure they don't need to retransmit the final
			// reliable message, but they don't do any other processing
			if (cl->state != CS_ZOMBIE)
			{
				cl->lastPacketTime = svs.time;  // don't timeout
				SV_ExecuteClientMessage(cl, msg);
			}
		}
		return;
	}

	// if we received a sequenced packet from an address we don't recognize,
	// send an out of band disconnect packet to it
	NET_OutOfBandPrint_Facade(NS_SERVER, from, "disconnect");
}

#endif