/*
 * Wolfenstein: Enemy Territory GPL Source Code
 * Copyright (C) 1999-2010 id Software LLC, a ZeniMax Media company.
 *
 * ET: Legacy
 * Copyright (C) 2012 Jan Simek <mail@etlegacy.com>
 *
 * This file is part of ET: Legacy - http://www.etlegacy.com
 *
 * ET: Legacy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ET: Legacy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ET: Legacy. If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, Wolfenstein: Enemy Territory GPL Source Code is also
 * subject to certain additional terms. You should have received a copy
 * of these additional terms immediately following the terms and conditions
 * of the GNU General Public License which accompanied the source code.
 * If not, please request a copy in writing from id Software at the address below.
 *
 * id Software LLC, c/o ZeniMax Media Inc., Suite 120, Rockville, Maryland 20850 USA.
 *
 * @file net_chan.c
*7Killer
*This file is a part of facade port systemn, its an arranged copy of net_chan.c
 */

#include "../qcommon/q_shared.h"
#include "qcommon.h"

#ifdef DEDICATED
/*

packet header
-------------
4   outgoing sequence.  high bit will be set if this is a fragmented message
[2  qport (only for client to server)]
[2  fragment start byte]
[2  fragment length. if < FRAGMENT_SIZE, this is the last fragment]

if the sequence number is -1, the packet should be handled as an out-of-band
message instead of as part of a netcon.

All fragments will have the same sequence numbers.

The qport field is a workaround for bad address translating routers that
sometimes remap the client's source port on a packet during gameplay.

If the base part of the net address matches and the qport matches, then the
channel matches even if the IP port differs.  The IP port should be updated
to the new value before sending out any replies.

*/



extern cvar_t *showpackets;
extern cvar_t *showdrop;
extern cvar_t *qport;



//=============================================================================

typedef struct packetQueue_s
{
	struct packetQueue_s *next;
	int length;
	byte *data;
	netadr_t to;
	int release;
} packetQueue_t;			//7killer TODO Move that into .h

packetQueue_t *packetQueue_F = NULL;

static void NET_QueuePacket_Facade(int length, const void *data, netadr_t to,
                            int offset)
{
	packetQueue_t *new, *next = packetQueue_F;

	if (offset > 999)
	{
		offset = 999;
	}

	new       = S_Malloc(sizeof(packetQueue_t));
	new->data = S_Malloc(length);
	Com_Memcpy(new->data, data, length);
	new->length  = length;
	new->to      = to;
	new->release = Sys_Milliseconds() + (int)((float)offset / com_timescale->value);
	new->next    = NULL;

	if (!packetQueue_F)
	{
		packetQueue_F = new;
		return;
	}
	while (next)
	{
		if (!next->next)
		{
			next->next = new;
			return;
		}
		next = next->next;
	}
}

void NET_FlushPacketQueue_Facade(void)
{
	packetQueue_t *last;
	int           now;

	while (packetQueue_F)
	{
		now = Sys_Milliseconds();
		if (packetQueue_F->release >= now)
		{
			break;
		}
		Sys_SendPacket_Facade(packetQueue_F->length, packetQueue_F->data,
		               packetQueue_F->to);
		last        = packetQueue_F;
		packetQueue_F = packetQueue_F->next;
		Z_Free(last->data);
		Z_Free(last);
	}
}

void NET_SendPacket_Facade(netsrc_t sock, int length, const void *data, netadr_t to)
{

	// sequenced packets are shown in netchan, so just show oob
	if (showpackets->integer && *(int *)data == -1)
	{
		Com_Printf("send facade packet %4i\n", length);
	}

	if (to.type == NA_LOOPBACK)
	{
		NET_SendLoopPacket(sock, length, data, to);
		return;
	}
	if (to.type == NA_BOT)
	{
		return;
	}
	if (to.type == NA_BAD)
	{
		return;
	}

	if (sock == NS_CLIENT && cl_packetdelay->integer > 0)
	{
		NET_QueuePacket_Facade(length, data, to, cl_packetdelay->integer);
	}
	else if (sock == NS_SERVER && sv_packetdelay->integer > 0)
	{
		NET_QueuePacket_Facade(length, data, to, sv_packetdelay->integer);
	}
	else
	{
		Sys_SendPacket_Facade(length, data, to);
	}
}

/*
===============
NET_OutOfBandPrint

Sends a text message in an out-of-band datagram
================
*/
void QDECL NET_OutOfBandPrint_Facade(netsrc_t sock, netadr_t adr, const char *format, ...)
{
	va_list argptr;
	char    string[MAX_MSGLEN];


	// set the header
	string[0] = -1;
	string[1] = -1;
	string[2] = -1;
	string[3] = -1;

	va_start(argptr, format);
	Q_vsnprintf(string + 4, sizeof(string) - 4, format, argptr);
	va_end(argptr);

	// send the datagram
	NET_SendPacket_Facade(sock, strlen(string), string, adr);
}

/*
===============
NET_OutOfBandPrint

Sends a data message in an out-of-band datagram (only used for "connect")
================
*/
void QDECL NET_OutOfBandData_Facade(netsrc_t sock, netadr_t adr, byte *format, int len)
{
	byte  string[MAX_MSGLEN * 2];
	int   i;
	msg_t mbuf;

	// set the header
	string[0] = 0xff;
	string[1] = 0xff;
	string[2] = 0xff;
	string[3] = 0xff;

	for (i = 0; i < len; i++)
	{
		string[i + 4] = format[i];
	}

	mbuf.data    = string;
	mbuf.cursize = len + 4;
	Huff_Compress(&mbuf, 12);
	// send the datagram
	NET_SendPacket_Facade(sock, mbuf.cursize, mbuf.data, adr);
}


#endif
