/*
 * Wolfenstein: Enemy Territory GPL Source Code
 * Copyright (C) 1999-2010 id Software LLC, a ZeniMax Media company.
 *
 * ET: Legacy
 * Copyright (C) 2012 Jan Simek <mail@etlegacy.com>
 *
 * This file is part of ET: Legacy - http://www.etlegacy.com
 *
 * ET: Legacy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ET: Legacy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ET: Legacy. If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, Wolfenstein: Enemy Territory GPL Source Code is also
 * subject to certain additional terms. You should have received a copy
 * of these additional terms immediately following the terms and conditions
 * of the GNU General Public License which accompanied the source code.
 * If not, please request a copy in writing from id Software at the address below.
 *
 * id Software LLC, c/o ZeniMax Media Inc., Suite 120, Rockville, Maryland 20850 USA.
 *
 * @file common_facade.c
 * @brief misc functions used in server facade port.
 *7Killer This files contains the functions need by the facade port , its a copy wiht some arrangement of the common.c file
 */

#include "q_shared.h"
#include "qcommon.h"
#include <setjmp.h>

#ifndef _WIN32
#include <netinet/in.h>
#include <sys/stat.h> // umask
#include <unistd.h> // getpid() - this is POSIX check MAC build
#else
#include <winsock.h>
#endif
#ifdef DEDICATED

#define MAX_NUM_ARGVS   50

#define MIN_DEDICATED_COMHUNKMEGS   1
#define MIN_COMHUNKMEGS             64
#define DEF_COMHUNKMEGS             128
#define DEF_COMZONEMEGS             24
#define DEF_COMHUNKMEGS_S           XSTRING(DEF_COMHUNKMEGS)
#define DEF_COMZONEMEGS_S           XSTRING(DEF_COMZONEMEGS)



#if idx64
int (*Q_VMftol)(void);
#elif id386
long (QDECL *Q_ftol)(float f);
int  (QDECL *Q_VMftol)(void);
void (QDECL *Q_SnapVector)(vec3_t vec);
#endif


//============================================================================

static char *rd_buffer;
static int  rd_buffersize;
static void (*rd_flush)(char *buffer);




/*
===================================================================

EVENTS AND JOURNALING

In addition to these events, .cfg files are also copied to the
journaled file
===================================================================
*/

#define MAX_PUSHED_EVENTS_F               1024
static int        com_pushedEventsHead_F = 0;
static int        com_pushedEventsTail_F = 0;
static sysEvent_t com_pushedEvents_F[MAX_PUSHED_EVENTS_F];

fileHandle_t        com_journalFile_F;        // events are written here
fileHandle_t        com_journalDataFile_F;    // config files are written here

extern cvar_t  *com_journal;
extern cvar_t  *com_dropsim; 
/*
=================
Com_InitJournaling
=================
*/
void Com_InitJournaling_Facade(void)
{
	Com_StartupVariable("journal_F");
	com_journal = Cvar_Get("journal_F", "0", CVAR_INIT);
	if (!com_journal->integer)
	{
		return;
	}

	if (com_journal->integer == 1)
	{
		Com_Printf("Journaling facade events\n");
		com_journalFile_F     = FS_FOpenFileWrite("journal_F.dat");
		com_journalDataFile_F = FS_FOpenFileWrite("journaldata_F.dat");
	}
	else if (com_journal->integer == 2)
	{
		Com_Printf("Replaying journaled facade events\n");
		FS_FOpenFileRead("journal_F.dat", &com_journalFile_F, qtrue);
		FS_FOpenFileRead("journaldata_F.dat", &com_journalDataFile_F, qtrue);
	}

	if (!com_journalFile_F || !com_journalDataFile_F)
	{
		Cvar_Set("com_journal_F", "0");
		com_journalFile_F     = 0;
		com_journalDataFile_F = 0;
		Com_Printf("Couldn't open journal facade files\n");
	}
}

/*
========================================================================

EVENT LOOP

========================================================================
*/

#define MAX_QUEUED_EVENTS_F  256
#define MASK_QUEUED_EVENTS_F (MAX_QUEUED_EVENTS_F - 1)

static sysEvent_t eventQueue_F[MAX_QUEUED_EVENTS_F];
static int        eventHead_F = 0;
static int        eventTail_F = 0;
static byte       sys_packetReceived_F[MAX_MSGLEN];

/*
================
Com_QueueEvent

A time of 0 will get the current time
Ptr should either be null, or point to a block of data that can
be freed by the game later.
================
*/
void Com_QueueEvent_Facade(int time, sysEventType_t type, int value, int value2, int ptrLength, void *ptr)
{
	sysEvent_t *ev;

	ev = &eventQueue_F[eventHead_F & MASK_QUEUED_EVENTS_F];

	if (eventHead_F - eventTail_F >= MAX_QUEUED_EVENTS_F)
	{
		Com_Printf("Com_QueueEvent_F: overflow\n");
		// we are discarding an event, but don't leak memory
		if (ev->evPtr)
		{
			Z_Free(ev->evPtr);
		}
		eventTail_F++;
	}

	eventHead_F++;

	if (time == 0)
	{
		time = Sys_Milliseconds();
	}

	ev->evTime      = time;
	ev->evType      = type;
	ev->evValue     = value;
	ev->evValue2    = value2;
	ev->evPtrLength = ptrLength;
	ev->evPtr       = ptr;
}

/*
================
Com_GetSystemEvent

================
*/
sysEvent_t Com_GetSystemEvent_Facade(void)
{
	sysEvent_t ev;
	msg_t      netmsg;
	netadr_t   adr;

	// return if we have data
	if (eventHead_F > eventTail_F)
	{
		eventTail_F++;
		return eventQueue_F[(eventTail_F - 1) & MASK_QUEUED_EVENTS_F];
	}
	
	// check for network packets
	MSG_Init(&netmsg, sys_packetReceived_F, sizeof(sys_packetReceived_F));
	if (Sys_GetPacket_Facade(&adr, &netmsg))
	{
		netadr_t *buf;
		int      len;

		// copy out to a seperate buffer for qeueing
		len  = sizeof(netadr_t) + netmsg.cursize;
		buf  = Z_Malloc(len);
		*buf = adr;
		memcpy(buf + 1, netmsg.data, netmsg.cursize);
		Com_QueueEvent_Facade(0, SE_PACKET, 0, 0, len, buf);
	}

	// return if we have data
	if (eventHead_F > eventTail_F)
	{
		eventTail_F++;
		return eventQueue_F[(eventTail_F - 1) & MASK_QUEUED_EVENTS_F];
	}

	// create an empty event to return
	Com_Memset(&ev, 0, sizeof(ev));
	ev.evTime = Sys_Milliseconds();

	return ev;
}

sysEvent_t  Com_GetRealEvent_Facade(void)
{
	int        r;
	sysEvent_t ev;

	// either get an event from the system or the journal file
	if (com_journal->integer == 2)
	{
		r = FS_Read(&ev, sizeof(ev), com_journalFile_F);
		if (r != sizeof(ev))
		{
			Com_Error(ERR_FATAL, "Error reading from journal facade file");
		}
		if (ev.evPtrLength)
		{
			ev.evPtr = Z_Malloc(ev.evPtrLength);
			r        = FS_Read(ev.evPtr, ev.evPtrLength, com_journalFile_F);
			if (r != ev.evPtrLength)
			{
				Com_Error(ERR_FATAL, "Error reading from journal facade file");
			}
		}
	}
	else
	{
		ev = Com_GetSystemEvent_Facade();

		// write the journal value out if needed
		if (com_journal->integer == 1)
		{
			r = FS_Write(&ev, sizeof(ev), com_journalFile_F);
			if (r != sizeof(ev))
			{
				Com_Error(ERR_FATAL, "Error writing to journal facade file");
			}
			if (ev.evPtrLength)
			{
				r = FS_Write(ev.evPtr, ev.evPtrLength, com_journalFile_F);
				if (r != ev.evPtrLength)
				{
					Com_Error(ERR_FATAL, "Error writing to journal facade file");
				}
			}
		}
	}

	return ev;
}

void Com_InitPushEvent_Facade(void)
{
	// clear the static buffer array
	// this requires SE_NONE to be accepted as a valid but NOP event
	Com_Memset(com_pushedEvents_F, 0, sizeof(com_pushedEvents_F));
	// reset counters while we are at it
	// beware: GetEvent might still return an SE_NONE from the buffer
	com_pushedEventsHead_F = 0;
	com_pushedEventsTail_F = 0;
}


/*
=================
Com_PushEvent
=================
*/
void Com_PushEvent_Facade(sysEvent_t *event)
{
	sysEvent_t *ev;
	static int printedWarning = 0;

	ev = &com_pushedEvents_F[com_pushedEventsHead_F & (MAX_PUSHED_EVENTS_F - 1)];

	if (com_pushedEventsHead_F - com_pushedEventsTail_F >= MAX_PUSHED_EVENTS_F)
	{

		// don't print the warning constantly, or it can give time for more...
		if (!printedWarning)
		{
			printedWarning = qtrue;
			Com_Printf("WARNING: Com_PushEvent_Facade overflow\n");
		}

		if (ev->evPtr)
		{
			Z_Free(ev->evPtr);
		}
		com_pushedEventsTail_F++;
	}
	else
	{
		printedWarning = qfalse;
	}

	*ev = *event;
	com_pushedEventsHead_F++;
}

/*
=================
Com_GetEvent
=================
*/
sysEvent_t  Com_GetEvent_Facade(void)
{
	if (com_pushedEventsHead_F > com_pushedEventsTail_F)
	{
		com_pushedEventsTail_F++;
		return com_pushedEvents_F[(com_pushedEventsTail_F - 1) & (MAX_PUSHED_EVENTS_F - 1)];
	}
	return Com_GetRealEvent_Facade();
}

/*
=================
Com_RunAndTimeServerPacket
=================
*/
void Com_RunAndTimeServerPacket_Facade(netadr_t *evFrom, msg_t *buf)
{
	int t1, t2, msec;

	t1 = 0;

	if (com_speeds->integer)
	{
		t1 = Sys_Milliseconds();
	}

	SV_PacketEvent_Facade(*evFrom, buf);

	if (com_speeds->integer)
	{
		t2   = Sys_Milliseconds();
		msec = t2 - t1;
		if (com_speeds->integer == 3)
		{
			Com_Printf("SV_PacketEvent_Facade time: %i\n", msec);
		}
	}
}

/*
=================
Com_EventLoop

Returns last event time
=================
*/

/*7Killer , as facade part require less events
I think a lot of them are useless, like client events 

*/
int Com_EventLoop_Facade(void)
{
	sysEvent_t ev;
	netadr_t   evFrom;
	byte       bufData[MAX_MSGLEN];
	msg_t      buf;

	MSG_Init(&buf, bufData, sizeof(bufData));

	while (1)
	{
		ev = Com_GetEvent_Facade();

		// if no more events are available
		if (ev.evType == SE_NONE)
		{
			// manually send packet events for the loopback channel
			while (NET_GetLoopPacket(NS_CLIENT, &evFrom, &buf))
			{
				CL_PacketEvent(evFrom, &buf);
			}

			while (NET_GetLoopPacket(NS_SERVER, &evFrom, &buf))
			{
				// if the server just shut down, flush the events
				if (com_sv_running->integer)
				{
					Com_RunAndTimeServerPacket_Facade(&evFrom, &buf);
				}
			}

			return ev.evTime;
		}


		switch (ev.evType)
		{
		default:
			// bk001129 - was ev.evTime
			Com_Error(ERR_FATAL, "Com_EventLoop_Facade: bad event type %i", ev.evType);
			break;
		case SE_NONE:
			break;
		case SE_KEY:
			CL_KeyEvent(ev.evValue, ev.evValue2, ev.evTime);
			break;
		case SE_CHAR:
			CL_CharEvent(ev.evValue);
			break;
		case SE_MOUSE:
			CL_MouseEvent(ev.evValue, ev.evValue2, ev.evTime);
			break;
		case SE_JOYSTICK_AXIS:
			CL_JoystickEvent(ev.evValue, ev.evValue2, ev.evTime);
			break;
		case SE_CONSOLE:
			Cbuf_AddText((char *)ev.evPtr);
			Cbuf_AddText("\n");
			break;
		case SE_PACKET:
			// this cvar allows simulation of connections that
			// drop a lot of packets.  Note that loopback connections
			// don't go through here at all.
			if (com_dropsim->value > 0)
			{
				static int seed;

				if (Q_random(&seed) < com_dropsim->value)
				{
					break;      // drop this packet
				}
			}

			evFrom      = *(netadr_t *)ev.evPtr;
			buf.cursize = ev.evPtrLength - sizeof(evFrom);

			// we must copy the contents of the message out, because
			// the event buffers are only large enough to hold the
			// exact payload, but channel messages need to be large
			// enough to hold fragment reassembly
			if ((unsigned)buf.cursize > buf.maxsize)
			{
				Com_Printf("Com_EventLoop_Facade: oversize packet\n");
				continue;
			}
			memcpy(buf.data, ( byte * )((netadr_t *)ev.evPtr + 1), buf.cursize);
			if (com_sv_running->integer)
			{
				Com_RunAndTimeServerPacket_Facade(&evFrom, &buf);
			}
			else
			{
				CL_PacketEvent(evFrom, &buf);
			}
			break;
		}

		// free any block data
		if (ev.evPtr)
		{
			Z_Free(ev.evPtr);
		}
	}

	return 0;   // never reached
}

/*
================
Com_Milliseconds

Can be used for profiling, but will be journaled accurately
================
*/
int Com_Milliseconds_Facade(void)
{
	sysEvent_t ev;

	// get events and push them until we get a null event with the current time
	do
	{

		ev = Com_GetRealEvent_Facade();
		if (ev.evType != SE_NONE)
		{
			Com_PushEvent_Facade(&ev);
		}
	}
	while (ev.evType != SE_NONE);

	return ev.evTime;
}
#endif
//============================================================================

